//---------------------------------------------------------------------------------------
//  FILE:   XComDownloadableContentInfo_BioticClass.uc                                    
//           
//	Use the X2DownloadableContentInfo class to specify unique mod behavior when the 
//  player creates a new campaign or loads a saved game.
//  
//---------------------------------------------------------------------------------------
//  Copyright (c) 2016 Firaxis Games, Inc. All rights reserved.
//---------------------------------------------------------------------------------------

class X2DownloadableContentInfo_BioticClassRedux extends X2DownloadableContentInfo;

/// <summary>
/// This method is run if the player loads a saved game that was created prior to this DLC / Mod being installed, and allows the 
/// DLC / Mod to perform custom processing in response. This will only be called once the first time a player loads a save that was
/// create without the content installed. Subsequent saves will record that the content was installed.
/// </summary>
static event OnLoadedSavedGame()
{

	UpdateStorage();

}

/// <summary>
/// Called when the player starts a new campaign while this DLC / Mod is installed
/// </summary>
static event InstallNewCampaign(XComGameState StartState)
{}


static event OnLoadedSavedGameToStrategy()
{

	UpdateStorage();

}

// ******** HANDLE UPDATING STORAGE ************* //
// This handles updating storage in order to create variations of various SMGs based on techs unlocked
static function UpdateStorage()
{
	local XComGameState NewGameState;
	local XComGameStateHistory History;
	local XComGameState_HeadquartersXCom XComHQ;
	local X2ItemTemplateManager ItemTemplateMgr;
	local X2ItemTemplate ItemTemplate;
	//local X2SchematicTemplate SchematicTemplate_MG, SchematicTemplate_BM;
	local XComGameState_Item NewItemState;

	History = `XCOMHISTORY;
	NewGameState = class'XComGameStateContext_ChangeContainer'.static.CreateChangeState("Updating HQ Storage to add SMGs");
	XComHQ = XComGameState_HeadquartersXCom(History.GetSingleGameStateObjectForClass(class'XComGameState_HeadquartersXCom'));
	XComHQ = XComGameState_HeadquartersXCom(NewGameState.CreateStateObject(class'XComGameState_HeadquartersXCom', XComHQ.ObjectID));
	NewGameState.AddStateObject(XComHQ);
	ItemTemplateMgr = class'X2ItemTemplateManager'.static.GetItemTemplateManager();

	//add Conventional SMG always
	//`Log("LW SMGPack : Updated Conventional SMG");
	ItemTemplate = ItemTemplateMgr.FindItemTemplate('RM_BioAmp_CV');
	if(ItemTemplate != none)
	{
		//`Log("LW SMGPack : Found SMG_CV item template");
		if (!XComHQ.HasItem(ItemTemplate))
		{
			//`Log("LW SMGPack : SMG_CV not found, adding to inventory");
			NewItemState = ItemTemplate.CreateInstanceFromTemplate(NewGameState);
			NewGameState.AddStateObject(NewItemState);
			XComHQ.AddItemToHQInventory(NewItemState);
			History.AddGameStateToHistory(NewGameState);
		} else {
			//`Log("LW SMGPack : SMG_CV found, skipping inventory add");
			History.CleanupPendingGameState(NewGameState);
		}

	}

	//schematics should be handled already, as the BuildItem UI draws from ItemTemplates, which are automatically loaded



}

static function bool AbilityTagExpandHandler(string InString, out string OutString)
{
	local name Type;
	local StateObjectReference UnitRef;
	local XComGameState_Unit UnitState;

	Type = name(InString);

	switch(Type)
	{
	case 'ReactiveBarrier':
		OutString = string(class'X2Ability_BioticGuardianSet'.default.ReactiveBarrier_Dmg_Adjust * 100);
		return true;
	case 'SAFEGUARD':
		OutString = string(class'X2Ability_BioticGuardianSet'.default.SafeguardPoints);
		return true;
	case 'BIOTICREGEN':
		OutString = string(class'X2Ability_BioticGuardianSet'.default.RegenPoints);
		return true;
	case 'BIOTIC_BARRIER':
		OutString = string(class'X2Ability_BioticGuardianSet'.default.BarrierPoints);
		return true;
	case 'BIOTICDEFENCE':
		OutString = string(class'X2Ability_BioticGuardianSet'.default.BIOTIC_DEFENCE);
		return true;
	case 'BIOTICTELEPORT':
		OutString = string(class'X2Ability_BioticGuardianSet'.default.Teleport_Range);
		return true;
	case 'Exposure_Rupture':
		OutString = string(class'X2Ability_BioticAdeptSet'.default.Exposure_Rupture);
		return true;
	case 'REAVE':
		OutString = string(class'X2Ability_BioticAdeptSet'.default.REAVE_BASE);
		return true;
	}
	return false;
}